(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define('jsonlint', ['exports'], factory) :
  (global = global || self, factory(global.jsonlint = {}))
}(this, function (exports) { 'use strict'

// Modified from https://github.com/rlidwka/jju/blob/master/lib/unicode.js

// This is autogenerated with esprima tools, see:
// https://github.com/ariya/esprima/blob/master/esprima.js

// eslint-disable-next-line no-unused-vars
const Uni = {
  isWhiteSpace: function isWhiteSpace (x) {
    // section 7.2, table 2
    return x === '\u0020' ||
        x === '\u00A0' ||
        x === '\uFEFF' || // <-- this is not a Unicode WS, only a JS one
        (x >= '\u0009' && x <= '\u000D') || // 9 A B C D

        // + whitespace characters from unicode, category Zs
        x === '\u1680' ||
        (x >= '\u2000' && x <= '\u200A') || // 0 1 2 3 4 5 6 7 8 9 A
        x === '\u2028' ||
        x === '\u2029' ||
        x === '\u202F' ||
        x === '\u205F' ||
        x === '\u3000'
  },
  isWhiteSpaceJSON: function isWhiteSpaceJSON (x) {
    return x === '\u0020' ||
        x === '\u0009' ||
        x === '\u000A' ||
        x === '\u000D'
  },
  isLineTerminator: function isLineTerminator (x) {
    // ok, here is the part when JSON is wrong
    // section 7.3, table 3
    return x === '\u000A' ||
        x === '\u000D' ||
        x === '\u2028' ||
        x === '\u2029'
  },
  isLineTerminatorJSON: function isLineTerminatorJSON (x) {
    return x === '\u000A' ||
        x === '\u000D'
  },
  isIdentifierStart: function isIdentifierStart (x) {
    return x === '$' ||
        x === '_' ||
        (x >= 'A' && x <= 'Z') ||
        (x >= 'a' && x <= 'z') ||
        (x >= '\u0080' && Uni.NonAsciiIdentifierStart.test(x))
  },
  isIdentifierPart: function isIdentifierPart (x) {
    return x === '$' ||
        x === '_' ||
        (x >= 'A' && x <= 'Z') ||
        (x >= 'a' && x <= 'z') ||
        (x >= '0' && x <= '9') || // <-- addition to Start
        (x >= '\u0080' && Uni.NonAsciiIdentifierPart.test(x))
  },
  // ECMAScript 5.1/Unicode v6.3.0 NonAsciiIdentifierStart:
  NonAsciiIdentifierStart: /[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0\u08A2-\u08AC\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0977\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA697\uA6A0-\uA6EF\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]/,
  // ECMAScript 5.1/Unicode v6.3.0 NonAsciiIdentifierPart:
  /* eslint-disable-next-line no-misleading-character-class */
  NonAsciiIdentifierPart: /[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0300-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u0483-\u0487\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u0591-\u05BD\u05BF\u05C1\u05C2\u05C4\u05C5\u05C7\u05D0-\u05EA\u05F0-\u05F2\u0610-\u061A\u0620-\u0669\u066E-\u06D3\u06D5-\u06DC\u06DF-\u06E8\u06EA-\u06FC\u06FF\u0710-\u074A\u074D-\u07B1\u07C0-\u07F5\u07FA\u0800-\u082D\u0840-\u085B\u08A0\u08A2-\u08AC\u08E4-\u08FE\u0900-\u0963\u0966-\u096F\u0971-\u0977\u0979-\u097F\u0981-\u0983\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BC-\u09C4\u09C7\u09C8\u09CB-\u09CE\u09D7\u09DC\u09DD\u09DF-\u09E3\u09E6-\u09F1\u0A01-\u0A03\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A3C\u0A3E-\u0A42\u0A47\u0A48\u0A4B-\u0A4D\u0A51\u0A59-\u0A5C\u0A5E\u0A66-\u0A75\u0A81-\u0A83\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABC-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0AD0\u0AE0-\u0AE3\u0AE6-\u0AEF\u0B01-\u0B03\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3C-\u0B44\u0B47\u0B48\u0B4B-\u0B4D\u0B56\u0B57\u0B5C\u0B5D\u0B5F-\u0B63\u0B66-\u0B6F\u0B71\u0B82\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD0\u0BD7\u0BE6-\u0BEF\u0C01-\u0C03\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55\u0C56\u0C58\u0C59\u0C60-\u0C63\u0C66-\u0C6F\u0C82\u0C83\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBC-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5\u0CD6\u0CDE\u0CE0-\u0CE3\u0CE6-\u0CEF\u0CF1\u0CF2\u0D02\u0D03\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D-\u0D44\u0D46-\u0D48\u0D4A-\u0D4E\u0D57\u0D60-\u0D63\u0D66-\u0D6F\u0D7A-\u0D7F\u0D82\u0D83\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0DCA\u0DCF-\u0DD4\u0DD6\u0DD8-\u0DDF\u0DF2\u0DF3\u0E01-\u0E3A\u0E40-\u0E4E\u0E50-\u0E59\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB9\u0EBB-\u0EBD\u0EC0-\u0EC4\u0EC6\u0EC8-\u0ECD\u0ED0-\u0ED9\u0EDC-\u0EDF\u0F00\u0F18\u0F19\u0F20-\u0F29\u0F35\u0F37\u0F39\u0F3E-\u0F47\u0F49-\u0F6C\u0F71-\u0F84\u0F86-\u0F97\u0F99-\u0FBC\u0FC6\u1000-\u1049\u1050-\u109D\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u135D-\u135F\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1714\u1720-\u1734\u1740-\u1753\u1760-\u176C\u176E-\u1770\u1772\u1773\u1780-\u17D3\u17D7\u17DC\u17DD\u17E0-\u17E9\u180B-\u180D\u1810-\u1819\u1820-\u1877\u1880-\u18AA\u18B0-\u18F5\u1900-\u191C\u1920-\u192B\u1930-\u193B\u1946-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u19D0-\u19D9\u1A00-\u1A1B\u1A20-\u1A5E\u1A60-\u1A7C\u1A7F-\u1A89\u1A90-\u1A99\u1AA7\u1B00-\u1B4B\u1B50-\u1B59\u1B6B-\u1B73\u1B80-\u1BF3\u1C00-\u1C37\u1C40-\u1C49\u1C4D-\u1C7D\u1CD0-\u1CD2\u1CD4-\u1CF6\u1D00-\u1DE6\u1DFC-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u200C\u200D\u203F\u2040\u2054\u2071\u207F\u2090-\u209C\u20D0-\u20DC\u20E1\u20E5-\u20F0\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D7F-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2DE0-\u2DFF\u2E2F\u3005-\u3007\u3021-\u302F\u3031-\u3035\u3038-\u303C\u3041-\u3096\u3099\u309A\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA62B\uA640-\uA66F\uA674-\uA67D\uA67F-\uA697\uA69F-\uA6F1\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA827\uA840-\uA873\uA880-\uA8C4\uA8D0-\uA8D9\uA8E0-\uA8F7\uA8FB\uA900-\uA92D\uA930-\uA953\uA960-\uA97C\uA980-\uA9C0\uA9CF-\uA9D9\uAA00-\uAA36\uAA40-\uAA4D\uAA50-\uAA59\uAA60-\uAA76\uAA7A\uAA7B\uAA80-\uAAC2\uAADB-\uAADD\uAAE0-\uAAEF\uAAF2-\uAAF6\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABEA\uABEC\uABED\uABF0-\uABF9\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE00-\uFE0F\uFE20-\uFE26\uFE33\uFE34\uFE4D-\uFE4F\uFE70-\uFE74\uFE76-\uFEFC\uFF10-\uFF19\uFF21-\uFF3A\uFF3F\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]/
}
/* globals Uni, getTexts */

// Modified from https://github.com/rlidwka/jju/blob/master/lib/parse.js

function isHexDigit (x) {
  return (x >= '0' && x <= '9') ||
      (x >= 'A' && x <= 'F') ||
      (x >= 'a' && x <= 'f')
}

function isOctDigit (x) {
  return x >= '0' && x <= '7'
}

function isDecDigit (x) {
  return x >= '0' && x <= '9'
}

function isBOM (x) {
  // This catches EFBBBF (UTF-8 BOM) because the buffer-to-string
  // conversion in `fs.readFileSync()` translates it to FEFF (UTF-16 BOM).
  return x.charCodeAt(0) === 0xFEFF
}

const unescapeMap = {
  '\'': '\'',
  '"': '"',
  '\\': '\\',
  b: '\b',
  f: '\f',
  n: '\n',
  r: '\r',
  t: '\t',
  v: '\v',
  '/': '/'
}

function parseInternal (input, options) {
  if (typeof input !== 'string' || !(input instanceof String)) {
    input = String(input)
  }

  const json5 = options.mode === 'json5'
  const ignoreBOM = options.ignoreBOM
  const ignoreComments = options.ignoreComments || options.mode === 'cjson' || json5
  const ignoreTrailingCommas = options.ignoreTrailingCommas || json5
  const allowSingleQuotedStrings = options.allowSingleQuotedStrings || json5
  const allowDuplicateObjectKeys = options.allowDuplicateObjectKeys
  const reviver = options.reviver
  const tokenize = options.tokenize
  const rawTokens = options.rawTokens
  const tokenLocations = options.tokenLocations
  const tokenPaths = options.tokenPaths

  const isLineTerminator = json5 ? Uni.isLineTerminator : Uni.isLineTerminatorJSON
  const isWhiteSpace = json5 ? Uni.isWhiteSpace : Uni.isWhiteSpaceJSON

  const inputLength = input.length
  let lineNumber = 0
  let lineStart = 0
  let position = 0

  const tokens = []
  let startToken
  let endToken
  let tokenPath

  if (tokenize) {
    let tokenOffset = null
    let tokenLine
    let tokenColumn
    startToken = function () {
      if (tokenOffset !== null) throw Error('internal error, token overlap')
      tokenLine = lineNumber + 1
      tokenColumn = position - lineStart + 1
      tokenOffset = position
    }
    endToken = function (type, value) {
      if (tokenOffset !== position) {
        const token = { type }
        if (rawTokens) {
          token.raw = input.substr(tokenOffset, position - tokenOffset)
        }
        if (value !== undefined) {
          token.value = value
        }
        if (tokenLocations) {
          token.location = {
            start: {
              column: tokenColumn,
              line: tokenLine,
              offset: tokenOffset
            }
          }
        }
        if (tokenPaths) {
          token.path = tokenPath.slice()
        }
        tokens.push(token)
      }
      tokenOffset = null
      return value
    }
    tokenPaths && (tokenPath = [])
  }

  function generateMessage () {
    let message
    if (position < inputLength) {
      const token = JSON.stringify(input[position])
      message = 'Unexpected token ' + token
    } else {
      message = 'Unexpected end of input'
    }
    return message
  }

  function createError (message) {
    const column = position - lineStart + 1
    ++lineNumber
    const texts = getTexts(message, input, position, lineNumber, column)
    const error = SyntaxError(texts.message)
    error.reason = message
    error.excerpt = texts.excerpt
    error.pointer = texts.pointer
    error.location = {
      start: {
        column,
        line: lineNumber,
        offset: position
      }
    }
    return error
  }

  function fail (message) {
    if (!message) {
      message = generateMessage()
    }
    const error = createError(message)
    throw error
  }

  function newLine (char) {
    // account for <cr><lf>
    if (char === '\r' && input[position] === '\n') {
      ++position
    }
    lineStart = position
    ++lineNumber
  }

  function parseGeneric () {
    if (position < inputLength) {
      startToken && startToken()
      const char = input[position++]
      if (char === '"' || (char === '\'' && allowSingleQuotedStrings)) {
        const string = parseString(char)
        endToken && endToken('literal', string)
        return string
      } else if (char === '{') {
        endToken && endToken('symbol', '{')
        return parseObject()
      } else if (char === '[') {
        endToken && endToken('symbol', '[')
        return parseArray()
      } else if (char === '-' || char === '.' || isDecDigit(char) ||
                 (json5 && (char === '+' || char === 'I' || char === 'N'))) {
        const number = parseNumber()
        endToken && endToken('literal', number)
        return number
      } else if (char === 'n') {
        parseKeyword('null')
        endToken && endToken('literal', null)
        return null
      } else if (char === 't') {
        parseKeyword('true')
        endToken && endToken('literal', true)
        return true
      } else if (char === 'f') {
        parseKeyword('false')
        endToken && endToken('literal', false)
        return false
      } else {
        --position
        endToken && endToken()
        return undefined
      }
    }
  }

  function parseKey () {
    let result
    if (position < inputLength) {
      startToken && startToken()
      const char = input[position++]
      if (char === '"' || (char === '\'' && allowSingleQuotedStrings)) {
        const string = parseString(char)
        endToken && endToken('literal', string)
        return string
      } else if (char === '{') {
        endToken && endToken('symbol', '{')
        return parseObject()
      } else if (char === '[') {
        endToken && endToken('symbol', '[')
        return parseArray()
      } else if (char === '.' || isDecDigit(char)) {
        const number = parseNumber(true)
        endToken && endToken('literal', number)
        return number
      } else if ((json5 && Uni.isIdentifierStart(char)) ||
                 (char === '\\' && input[position] === 'u')) {
        const rollback = position - 1
        result = parseIdentifier()
        if (result === undefined) {
          position = rollback
          endToken && endToken()
          return undefined
        } else {
          endToken && endToken('literal', result)
          return result
        }
      } else {
        --position
        endToken && endToken()
        return undefined
      }
    }
  }

  function skipBOM () {
    if (isBOM(input)) {
      startToken && startToken()
      ++position
      endToken && endToken('bom')
    }
  }

  function skipWhiteSpace () {
    let insideWhiteSpace
    function startWhiteSpace () {
      if (!insideWhiteSpace) {
        insideWhiteSpace = true
        --position
        startToken()
        ++position
      }
    }
    function endWhiteSpace () {
      if (insideWhiteSpace) {
        insideWhiteSpace = false
        endToken('whitespace')
      }
    }
    while (position < inputLength) {
      const char = input[position++]
      if (isLineTerminator(char)) {
        startToken && startWhiteSpace()
        newLine(char)
      } else if (isWhiteSpace(char)) {
        startToken && startWhiteSpace()
      } else if (char === '/' && ignoreComments &&
                 (input[position] === '/' || input[position] === '*')) {
        if (startToken) {
          --position
          endWhiteSpace()
          startToken()
          ++position
        }
        skipComment(input[position++] === '*')
        endToken && endToken('comment')
      } else {
        --position
        break
      }
    }
    endToken && endWhiteSpace()
  }

  function skipComment (multiLine) {
    while (position < inputLength) {
      const char = input[position++]
      if (isLineTerminator(char)) {
        if (!multiLine) {
          // let parent function deal with newline
          --position
          return
        }
        newLine(char)
      } else if (char === '*' && multiLine) {
        if (input[position] === '/') {
          ++position
          return
        }
      } else {
        // nothing
      }
    }
    if (multiLine) {
      fail('Unclosed multiline comment')
    }
  }

  function parseKeyword (keyword) {
    // keyword[0] is not checked because it was checked earlier
    const startPosition = position
    for (let i = 1, keywordLength = keyword.length; i < keywordLength; ++i) {
      if (position >= inputLength || keyword[i] !== input[position]) {
        position = startPosition - 1
        fail()
      }
      ++position
    }
  }

  function parseObject () {
    const result = {}
    const emptyObject = {}
    let isNotEmpty = false

    while (position < inputLength) {
      skipWhiteSpace()
      const key = parseKey()
      if (allowDuplicateObjectKeys === false && result[key]) {
        fail('Duplicate key: "' + key + '"')
      }
      skipWhiteSpace()
      startToken && startToken()
      let char = input[position++]
      endToken && endToken('symbol', char)
      if (char === '}' && key === undefined) {
        if (!ignoreTrailingCommas && isNotEmpty) {
          --position
          fail('Trailing comma in object')
        }
        return result
      } else if (char === ':' && key !== undefined) {
        skipWhiteSpace()
        tokenPath && tokenPath.push(key)
        let value = parseGeneric()
        tokenPath && tokenPath.pop()

        if (value === undefined) fail('No value found for key "' + key + '"')
        if (typeof key !== 'string') {
          if (!json5 || typeof key !== 'number') {
            fail('Wrong key type: "' + key + '"')
          }
        }

        if (key in emptyObject || emptyObject[key] != null) {
          // silently ignore it
        } else {
          if (reviver) {
            value = reviver(key, value)
          }
          if (value !== undefined) {
            isNotEmpty = true
            result[key] = value
          }
        }

        skipWhiteSpace()
        startToken && startToken()
        char = input[position++]
        endToken && endToken('symbol', char)
        if (char === ',') {
          continue
        } else if (char === '}') {
          return result
        } else {
          fail()
        }
      } else {
        --position
        fail()
      }
    }

    fail()
  }

  function parseArray () {
    const result = []
    while (position < inputLength) {
      skipWhiteSpace()
      tokenPath && tokenPath.push(result.length)
      let item = parseGeneric()
      tokenPath && tokenPath.pop()
      skipWhiteSpace()
      startToken && startToken()
      const char = input[position++]
      endToken && endToken('symbol', char)
      if (item !== undefined) {
        if (reviver) {
          item = reviver(String(result.length), item)
        }
        if (item === undefined) {
          ++result.length
          item = true // hack for check below, not included into result
        } else {
          result.push(item)
        }
      }

      if (char === ',') {
        if (item === undefined) {
          fail('Elisions are not supported')
        }
      } else if (char === ']') {
        if (!ignoreTrailingCommas && item === undefined && result.length) {
          --position
          fail('Trailing comma in array')
        }
        return result
      } else {
        --position
        fail()
      }
    }
  }

  function parseNumber () {
    // rewind because we don't know first char
    --position

    let start = position
    let char = input[position++]
    const toNumber = function (isOctal) {
      const string = input.substr(start, position - start)
      let result

      if (isOctal) {
        result = parseInt(string.replace(/^0o?/, ''), 8)
      } else {
        result = Number(string)
      }

      if (Number.isNaN(result)) {
        --position
        fail('Bad numeric literal - "' + input.substr(start, position - start + 1) + '"')
      } else if (!json5 && !string.match(/^-?(0|[1-9][0-9]*)(\.[0-9]+)?(e[+-]?[0-9]+)?$/i)) {
        // additional restrictions imposed by json
        --position
        fail('Non-json numeric literal - "' + input.substr(start, position - start + 1) + '"')
      } else {
        return result
      }
    }

    // ex: -5982475.249875e+29384
    //     ^ skipping this
    if (char === '-' || (char === '+' && json5)) {
      char = input[position++]
    }

    if (char === 'N' && json5) {
      parseKeyword('NaN')
      return NaN
    }

    if (char === 'I' && json5) {
      parseKeyword('Infinity')
      // returning +inf or -inf
      return toNumber()
    }

    if (char >= '1' && char <= '9') {
      // ex: -5982475.249875e+29384
      //        ^^^ skipping these
      while (position < inputLength && isDecDigit(input[position])) {
        ++position
      }
      char = input[position++]
    }

    // special case for leading zero: 0.123456
    if (char === '0') {
      char = input[position++]

      //             new syntax, "0o777"           old syntax, "0777"
      const isOctal = char === 'o' || char === 'O' || isOctDigit(char)
      const isHex = char === 'x' || char === 'X'

      if (json5 && (isOctal || isHex)) {
        while (position < inputLength &&
               (isHex ? isHexDigit : isOctDigit)(input[position])) {
          ++position
        }

        let sign = 1
        if (input[start] === '-') {
          sign = -1
          ++start
        } else if (input[start] === '+') {
          ++start
        }

        return sign * toNumber(isOctal)
      }
    }

    if (char === '.') {
      // ex: -5982475.249875e+29384
      //                ^^^ skipping these
      while (position < inputLength && isDecDigit(input[position])) {
        ++position
      }
      char = input[position++]
    }

    if (char === 'e' || char === 'E') {
      char = input[position++]
      if (char === '-' || char === '+') {
        ++position
      }
      // ex: -5982475.249875e+29384
      //                       ^^^ skipping these
      while (position < inputLength && isDecDigit(input[position])) {
        ++position
      }
      char = input[position++]
    }

    // we have char in the buffer, so count for it
    --position
    return toNumber()
  }

  function parseIdentifier () {
    // rewind because we don't know first char
    --position

    let result = ''
    while (position < inputLength) {
      let char = input[position++]
      if (char === '\\' &&
          input[position] === 'u' &&
          isHexDigit(input[position + 1]) &&
          isHexDigit(input[position + 2]) &&
          isHexDigit(input[position + 3]) &&
          isHexDigit(input[position + 4])) {
        // UnicodeEscapeSequence
        char = String.fromCharCode(parseInt(input.substr(position + 1, 4), 16))
        position += 5
      }

      if (result.length) {
        // identifier started
        if (Uni.isIdentifierPart(char)) {
          result += char
        } else {
          --position
          return result
        }
      } else {
        if (Uni.isIdentifierStart(char)) {
          result += char
        } else {
          return undefined
        }
      }
    }

    fail()
  }

  function parseString (endChar) {
    // 7.8.4 of ES262 spec
    let result = ''
    while (position < inputLength) {
      let char = input[position++]
      if (char === endChar) {
        return result
      } else if (char === '\\') {
        if (position >= inputLength) {
          fail()
        }
        char = input[position++]
        if (unescapeMap[char] && (json5 || (char !== 'v' && (char !== "'" || allowSingleQuotedStrings)))) {
          result += unescapeMap[char]
        } else if (json5 && isLineTerminator(char)) {
          // line continuation
          newLine(char)
        } else if (char === 'u' || (char === 'x' && json5)) {
          // unicode/character escape sequence
          const count = char === 'u' ? 4 : 2
          // validation for \uXXXX
          for (let i = 0; i < count; ++i) {
            if (position >= inputLength) {
              fail()
            }
            if (!isHexDigit(input[position])) {
              fail('Bad escape sequence')
            }
            position++
          }
          result += String.fromCharCode(parseInt(input.substr(position - count, count), 16))
        } else if (json5 && isOctDigit(char)) {
          let digits
          if (char < '4' && isOctDigit(input[position]) && isOctDigit(input[position + 1])) {
            // three-digit octal
            digits = 3
          } else if (isOctDigit(input[position])) {
            // two-digit octal
            digits = 2
          } else {
            digits = 1
          }
          position += digits - 1
          result += String.fromCharCode(parseInt(input.substr(position - digits, digits), 8))
        } else if (json5) {
          // \X -> x
          result += char
        } else {
          --position
          fail()
        }
      } else if (isLineTerminator(char)) {
        fail()
      } else {
        if (!json5 && char.charCodeAt(0) < 32) {
          --position
          fail('Unexpected control character')
        }
        // SourceCharacter but not one of " or \ or LineTerminator
        result += char
      }
    }

    fail()
  }

  if (ignoreBOM) {
    skipBOM()
  }
  skipWhiteSpace()
  let returnValue = parseGeneric()
  if (returnValue !== undefined || position < inputLength) {
    skipWhiteSpace()
    if (position >= inputLength) {
      if (reviver) {
        returnValue = reviver('', returnValue)
      }
      return tokenize ? tokens : returnValue
    } else {
      fail()
    }
  } else {
    if (position) {
      fail('No data, only a whitespace')
    } else {
      fail('No data, empty input')
    }
  }
}

// eslint-disable-next-line no-unused-vars
function parseCustom (input, options) {
  if (typeof options === 'function') {
    options = {
      reviver: options
    }
  } else if (!options) {
    options = {}
  }
  return parseInternal(input, options)
}

// eslint-disable-next-line no-unused-vars
function tokenize (input, options) {
  if (!options) {
    options = {}
  }
  // As long as this is the single modification, this is easier than cloning.
  // (Once Node.js 6 is dropped, this can be replaced by object destructuring.)
  const oldTokenize = options.tokenize
  options.tokenize = true
  const tokens = parseInternal(input, options)
  options.tokenize = oldTokenize
  return tokens
}
function escapePointerToken (token) {
  return token
    .toString()
    .replace(/~/g, '~0')
    .replace(/\//g, '~1')
}

// eslint-disable-next-line no-unused-vars
function pathToPointer (tokens) {
  if (tokens.length === 0) {
    return ''
  }
  return '/' + tokens
    .map(escapePointerToken)
    .join('/')
}

function unescapePointerToken (token) {
  return token
    .replace(/~1/g, '/')
    .replace(/~0/g, '~')
}

// eslint-disable-next-line no-unused-vars
function pointerToPath (pointer) {
  if (pointer === '') {
    return []
  }
  if (pointer[0] !== '/') {
    throw new Error('Missing initial "/" in the reference')
  }
  return pointer
    .substr(1)
    .split('/')
    .map(unescapePointerToken)
}
function getLineAndColumn (input, offset) {
  const lines = input
    .substr(0, offset)
    .split(/\r?\n/)
  const line = lines.length
  const column = lines[line - 1].length + 1
  return {
    line,
    column
  }
}

function getOffset (input, line, column) {
  if (line > 1) {
    const breaks = /\r?\n/g
    let match
    while ((match = breaks.exec(input))) {
      if (--line === 1) {
        return match.index + column
      }
    }
  }
  return column - 1
}

function pastInput (input, offset) {
  const start = Math.max(0, offset - 20)
  const previous = input.substr(start, offset - start)
  return (offset > 20 ? '...' : '') + previous.replace(/\r?\n/g, '')
}

function upcomingInput (input, offset) {
  let start = Math.max(0, offset - 20)
  start += offset - start
  const rest = input.length - start
  const next = input.substr(start, Math.min(20, rest))
  return next.replace(/\r?\n/g, '') + (rest > 20 ? '...' : '')
}

function getPositionContext (input, offset) {
  const past = pastInput(input, offset)
  const upcoming = upcomingInput(input, offset)
  const pointer = new Array(past.length + 1).join('-') + '^'
  return {
    excerpt: past + upcoming,
    pointer
  }
}

function getReason (error) {
  let message = error.message
    .replace('JSON.parse: ', '') // SpiderMonkey
    .replace('JSON Parse error: ', '') // SquirrelFish
  const firstCharacter = message.charAt(0)
  if (firstCharacter >= 'a') {
    message = firstCharacter.toUpperCase() + message.substr(1)
  }
  return message
}

function getLocationOnV8 (input, reason) {
  const match = / in JSON at position (\d+)$/.exec(reason)
  if (match) {
    const offset = +match[1]
    const location = getLineAndColumn(input, offset)
    return {
      offset,
      line: location.line,
      column: location.column,
      reason: reason.substr(0, match.index)
    }
  }
}

function checkUnexpectedEndOnV8 (input, reason) {
  const match = / end of JSON input$/.exec(reason)
  if (match) {
    const offset = input.length
    const location = getLineAndColumn(input, offset)
    return {
      offset,
      line: location.line,
      column: location.column,
      reason: reason.substr(0, match.index + 4)
    }
  }
}

function getLocationOnSpiderMonkey (input, reason) {
  const match = / at line (\d+) column (\d+) of the JSON data$/.exec(reason)
  if (match) {
    const line = +match[1]
    const column = +match[2]
    const offset = getOffset(input, line, column) // eslint-disable-line no-undef
    return {
      offset,
      line,
      column,
      reason: reason.substr(0, match.index)
    }
  }
}

function getTexts (reason, input, offset, line, column) {
  const position = getPositionContext(input, offset)
  const excerpt = position.excerpt
  let message, pointer
  if (typeof line === 'number') {
    pointer = position.pointer
    message = 'Parse error on line ' + line + ', column ' +
      column + ':\n' + excerpt + '\n' + pointer + '\n' + reason
  } else {
    message = 'Parse error in JSON input:\n' + excerpt + '\n' + reason
  }
  return {
    message,
    excerpt,
    pointer
  }
}

function improveNativeError (input, error) {
  let reason = getReason(error)
  const location = getLocationOnV8(input, reason) ||
    checkUnexpectedEndOnV8(input, reason) ||
    getLocationOnSpiderMonkey(input, reason)
  let offset, line, column
  if (location) {
    offset = location.offset
    line = location.line
    column = location.column
    reason = location.reason
  } else {
    offset = 0
  }
  error.reason = reason
  const texts = getTexts(reason, input, offset, line, column)
  error.message = texts.message
  error.excerpt = texts.excerpt
  if (texts.pointer) {
    error.pointer = texts.pointer
    error.location = {
      start: {
        column,
        line,
        offset
      }
    }
  }
  return error
}

// eslint-disable-next-line no-unused-vars
function parseNative (input, reviver) {
  try {
    return JSON.parse(input, reviver)
  } catch (error) {
    const newError = improveNativeError(input, error)
    if (error.location) throw newError
    // If the native error didn't contain location, parse once more
    // by the custom parser, which always provides the error location.
    return parseCustom (input, reviver)
  }
}
/* globals navigator, process, parseCustom, parseNative */

const isSafari = typeof navigator !== 'undefined' && /Safari/.test(navigator.userAgent) && /Apple Computer/.test(navigator.vendor)
const oldNode = typeof process !== 'undefined' && process.version.startsWith('v4.')

function needsCustomParser (options) {
  return options.ignoreBOM || options.ignoreComments || options.ignoreTrailingCommas ||
  options.allowSingleQuotedStrings || options.allowDuplicateObjectKeys === false ||
  options.mode === 'cjson' || options.mode === 'json5' || isSafari || oldNode
}

function getReviver (options) {
  if (typeof options === 'function') {
    return options
  } else if (options) {
    return options.reviver
  }
}

// eslint-disable-next-line no-unused-vars
function parse (input, options) {
  options || (options = {})
  return needsCustomParser(options)
    ? parseCustom(input, options)
    : parseNative(input, getReviver(options))
}

  exports.parse = parse
  exports.tokenize = tokenize
  exports.pathToPointer = pathToPointer
  exports.pointerToPath = pointerToPath

  exports.parseNative = parseNative
  exports.parseCustom = parseCustom
  exports.getErrorTexts = getTexts

  Object.defineProperty(exports, '__esModule', { value: true })
}));
