import * as THREE from "three";
import { OrbitControls } from "three/OrbitControls.js";
import { GLTFLoader } from "three/GLTFLoader.js";
import { DRACOLoader } from "three/DRACOLoader.js";

// NOTE: stopRotate must be out of the anonymous function
const
    stopRotate = new CustomEvent("stopRotate"),
    loadBall = new CustomEvent("loadBall")
;

(function () {
    "use strict";

    let
        ball = null,
        autoRotate = true,
        timeoutId = null
    ;

    const
        url_model = "./models/ball_draco.glb",
        dev = false,
        id = "main_ball"
    ;

    addEventListener("load", function () {
        // scene
        const
            s1 = document.getElementById("container_ball"),
            camera = new THREE.PerspectiveCamera(30, 1, 0.1, 100),
            scene = new THREE.Scene()
        ;

        camera.position.set(0, 0, 1.35);

        // renderer
        const renderer = new THREE.WebGLRenderer({
            antialias:true,
            alpha:true,
            powerPreference: "high-performance"
        });
        renderer.setPixelRatio(Math.min(window.devicePixelRatio || 1, 2));

        renderer.domElement.id = id;
        renderer.domElement.classList.add(id);
        s1.appendChild(renderer.domElement);

        // controls
        const controls = new OrbitControls(camera, renderer.domElement);
        controls.enableDamping = true;
        controls.dampingFactor = 0.07;

        controls.enableRotate = true;
        controls.enableZoom = false;
        controls.enablePan = true;

        // homogenous lights
        const ambient = new THREE.AmbientLight(0xffffff, 2);
        scene.add(ambient);

        const hemi = new THREE.HemisphereLight(0xffffff, 0x888888, 1.8);
        scene.add(hemi);

        const dir1 = new THREE.DirectionalLight(0xff0000, 0.4);
        dir1.position.set(5, 10, 7.5);
        scene.add(dir1);

        const dir2 = new THREE.DirectionalLight(0x0000ff, 0.4);
        dir2.position.set(-5, -5, -5);
        scene.add(dir2);

        const loader = new GLTFLoader();

        const dracoLoader = new DRACOLoader();
        dracoLoader.setDecoderPath("https://www.gstatic.com/draco/versioned/decoders/1.5.7/");

        loader.setDRACOLoader(dracoLoader);

        loader.load(url_model,
            (gltf) => {
                ball = gltf.scene;
                ball.position.set(0, 0, 0);
                scene.add(ball);

                renderer.render(scene, camera);

                requestAnimationFrame(() => {
                    document.dispatchEvent(loadBall);
                    if (dev) console.log("Modelo cargado ✔");
                });

            },
            (xhr) => {
                if (xhr.loaded && xhr.total) {
                    if (dev) console.log("Cargando: " + Math.round(xhr.loaded / xhr.total * 100) + "%");
                }
            },
            (err) => {
                console.error(err);
                console.log("Error cargando GLB — ver consola");
            }
        );

        controls.addEventListener("start", () => {
            autoRotate = false;
            document.dispatchEvent(stopRotate);
        });

        controls.addEventListener("end", () => {
            if (timeoutId) clearTimeout(timeoutId);
            timeoutId = setTimeout(() => {
                autoRotate = true;
            }, 5000);
        });

        function animate() {
            requestAnimationFrame(animate);

            if (ball && autoRotate) ball.rotation.z -= 0.005;

            controls.update();
            renderer.render(scene, camera);
        }
        animate();

        function updateCameraAspect() {
            const w = renderer.domElement.clientWidth;
            const h = renderer.domElement.clientHeight;

            renderer.setSize(w, h, false);
            camera.aspect = w / h;
            camera.updateProjectionMatrix();
        }

        updateCameraAspect();
        window.addEventListener("resize", updateCameraAspect);
    });
}());
