(function () {
    "use strict";

    const
        DEFAULT_TOTAL_PAIRS = 6,
        DEFAULT_TIMER_DURATION = 30000,
        DEFAULT_CARD_RANGE = { min: 1, max: 9 },
        FLIP_BACK_DELAY = 1000
    ;

    const state = {
        config: {
            start_button: null,
            cover_elements: [],
            card_container: null,
            timer_display: null,
            matches_display: null,
            reload_button: null,
            inactive_class: "inactive",
            active_class: "active",
            matched_class: "matched",
            hidden_class: "hide",
            template_id: "template_card_passion",
            total_pairs: DEFAULT_TOTAL_PAIRS,
            card_range: { ...DEFAULT_CARD_RANGE }
        },
        timer_id: null,
        timer_duration: DEFAULT_TIMER_DURATION,
        timer_remaining: DEFAULT_TIMER_DURATION,
        random_cards: [],
        bound_start_handler: null,
        bound_card_handler: null,
        bound_reload_handler: null,
        selected_cards: [],
        matched_pairs: 0,
        is_board_locked: false,
        is_game_active: false,
        has_dispatched_result: false,
        base_timer_duration: DEFAULT_TIMER_DURATION
    };

    const utils = {
        random_int(min, max) {
            return Math.floor(Math.random() * (max - min + 1)) + min;
        },

        shuffle(array) {
            for (let i = array.length - 1; i > 0; i--) {
                const j = Math.floor(Math.random() * (i + 1));
                [array[i], array[j]] = [array[j], array[i]];
            }
            return array;
        },

        format_time(ms) {
            const
                total_seconds = Math.max(0, Math.floor(ms / 1000)),
                minutes = String(Math.floor(total_seconds / 60)).padStart(2, "0"),
                seconds = String(total_seconds % 60).padStart(2, "0")
            ;
            return `${minutes}:${seconds}`;
        },

        parse_timer_value(display, fallback) {
            if (!display) return fallback;

            const
                raw = (display.textContent || "").trim(),
                raw_seconds = parseInt(raw, 10)
            ;

            if (!raw) return fallback;

            if (raw.includes(":")) {
                const parts = raw.split(":").map(part => part.trim());
                if (parts.length !== 2) return fallback;

                const
                    minutes = parseInt(parts[0], 10),
                    seconds = parseInt(parts[1], 10),
                    total_seconds = minutes * 60 + seconds
                ;

                if (Number.isNaN(minutes) || Number.isNaN(seconds)) return fallback;

                return total_seconds > 0 ? total_seconds * 1000 : fallback;
            }

            if (Number.isNaN(raw_seconds)) return fallback;

            return raw_seconds > 0 ? raw_seconds * 1000 : fallback;
        },

        update_timer_display(ms) {
            if (!state.config.timer_display) return;
            state.config.timer_display.textContent = utils.format_time(ms);
        },

        update_matches_display() {
            if (!state.config.matches_display) return;
            state.config.matches_display.textContent = `${state.matched_pairs}`;
        },

        set_reload_button_hidden(is_hidden) {
            const
                { reload_button, hidden_class } = state.config
            ;
            if (!reload_button || !hidden_class) return;
            reload_button.classList.toggle(hidden_class, Boolean(is_hidden));
        },

        build_deck() {
            const
                { total_pairs, card_range } = state.config,
                unique = new Set(),
                min = card_range.min,
                max = card_range.max
            ;

            while (unique.size < total_pairs) {
                unique.add(utils.random_int(min, max));
            }

            const
                deck = Array.from(unique),
                doubled_deck = deck.concat(deck)
            ;
            return utils.shuffle(doubled_deck);
        },

        render_cards() {
            const
                container = state.config.card_container,
                template = document.getElementById(state.config.template_id)
            ;
            if (!container || !template) return;

            container.innerHTML = "";

            const
                deck = utils.build_deck(),
                fragment = document.createDocumentFragment()
            ;

            for (const id of deck) {
                const
                    card_fragment = template.content.cloneNode(true),
                    card = card_fragment.querySelector(".card_passion"),
                    img_back = card_fragment.querySelector(".img_back")
                ;

                if (card) {
                    card.dataset.id = id;
                    card.classList.remove(
                        state.config.active_class,
                        state.config.matched_class
                    );
                    card.dataset.matched = "false";
                }

                if (img_back) {
                    img_back.src = `img/passion/${id}.webp`;
                    img_back.alt = `Card ${id}`;
                }

                fragment.appendChild(card_fragment);
            }

            container.appendChild(fragment);
            state.random_cards = deck;
        },

        hide_cover_elements() {
            const { cover_elements, inactive_class } = state.config;
            for (const element of cover_elements) {
                if (!element) continue;
                element.classList.add(inactive_class);
            }
        },

        bind_start_button() {
            const { start_button } = state.config;
            if (!start_button) return;

            if (state.bound_start_handler) {
                start_button.removeEventListener("click", state.bound_start_handler);
            }

            state.bound_start_handler = function handle_start() {
                const { reload_button, hidden_class } = state.config;
                if (reload_button && hidden_class) {
                    reload_button.classList.remove(hidden_class);
                }
                utils.hide_cover_elements();
                PassionGame.start_game();
            };

            start_button.addEventListener("click", state.bound_start_handler);
        },

        bind_card_events() {
            const container = state.config.card_container;
            if (!container) return;

            if (state.bound_card_handler) {
                container.removeEventListener("click", state.bound_card_handler);
            }

            state.bound_card_handler = function handle_card_click(event) {
                const trigger = event.target.closest(".btn_front");
                if (!trigger) return;

                const card = trigger.closest(".card_passion");
                if (!card || !container.contains(card)) return;

                PassionGame.select_card(card);
            };

            container.addEventListener("click", state.bound_card_handler);
        },

        bind_reload_button() {
            const { reload_button } = state.config;
            if (!reload_button) return;

            if (state.bound_reload_handler) {
                reload_button.removeEventListener("click", state.bound_reload_handler);
            }

            state.bound_reload_handler = function handle_reload(event) {
                event.preventDefault();
                PassionGame.restart_game();
            };

            reload_button.addEventListener("click", state.bound_reload_handler);
        }
    };

    const PassionGame = {
        init(options = {}) {
            state.config = {
                ...state.config,
                start_button: options.start_button || null,
                cover_elements: Array.isArray(options.cover_elements) ? options.cover_elements.filter(Boolean) : [],
                card_container: options.card_container || null,
                timer_display: options.timer_display || null,
                matches_display: options.matches_display || null,
                reload_button: options.reload_button || null,
                hidden_class: (typeof options.hidden_class === "string" && options.hidden_class.length > 0)
                    ? options.hidden_class
                    : state.config.hidden_class,
                inactive_class: options.inactive_class || state.config.inactive_class,
                active_class: options.active_class || state.config.active_class,
                matched_class: options.matched_class || state.config.matched_class,
                template_id: options.template_id || state.config.template_id,
                total_pairs: Number.isFinite(options.total_pairs) && options.total_pairs > 0
                    ? Math.min(options.total_pairs, (state.config.card_range.max - state.config.card_range.min) + 1)
                    : state.config.total_pairs,
                card_range: options.card_range
                    ? {
                        min: Number.isFinite(options.card_range.min) ? options.card_range.min : DEFAULT_CARD_RANGE.min,
                        max: Number.isFinite(options.card_range.max) ? options.card_range.max : DEFAULT_CARD_RANGE.max
                    }
                    : { ...state.config.card_range }
            };

            if (typeof options.total_pairs === "number" && options.total_pairs > 0) {
                state.config.total_pairs = options.total_pairs;
            }

            if (options.card_range) {
                const
                    min = Number.isFinite(options.card_range.min) ? options.card_range.min : DEFAULT_CARD_RANGE.min,
                    max = Number.isFinite(options.card_range.max) ? options.card_range.max : DEFAULT_CARD_RANGE.max
                ;
                state.config.card_range = { min, max };
            }

            const
                parsed_duration = utils.parse_timer_value(state.config.timer_display, DEFAULT_TIMER_DURATION),
                base_duration = typeof options.timer_duration === "number" && options.timer_duration > 0
                    ? options.timer_duration
                    : parsed_duration
            ;

            state.base_timer_duration = base_duration;
            state.timer_duration = state.base_timer_duration;
            state.timer_remaining = state.base_timer_duration;
            state.selected_cards = [];
            state.matched_pairs = 0;
            state.is_board_locked = false;
            state.is_game_active = false;
            state.has_dispatched_result = false;

            utils.set_reload_button_hidden(false);
            utils.update_timer_display(state.timer_remaining);
            utils.update_matches_display();
            utils.render_cards();
            utils.bind_start_button();
            utils.bind_card_events();
            utils.bind_reload_button();
        },

        start_game() {
            utils.render_cards();
            state.selected_cards = [];
            state.matched_pairs = 0;
            state.is_board_locked = false;
            state.is_game_active = true;
            state.has_dispatched_result = false;
            state.timer_duration = state.base_timer_duration;
            state.timer_remaining = state.base_timer_duration;
            utils.set_reload_button_hidden(false);
            utils.update_timer_display(state.timer_remaining);
            utils.update_matches_display();
            PassionGame.start_timer();
        },

        start_timer() {
            PassionGame.stop_timer();

            utils.update_timer_display(state.timer_remaining);

            if (state.timer_remaining <= 0) {
                PassionGame.handle_defeat();
                return;
            }

            state.timer_id = window.setInterval(() => {
                if (!state.is_game_active) {
                    PassionGame.stop_timer();
                    return;
                }

                state.timer_remaining -= 1000;

                if (state.timer_remaining <= 0) {
                    state.timer_remaining = 0;
                    utils.update_timer_display(state.timer_remaining);
                    PassionGame.stop_timer();
                    PassionGame.handle_defeat();
                    return;
                }

                utils.update_timer_display(state.timer_remaining);
            }, 1000);
        },

        stop_timer() {
            if (state.timer_id !== null) {
                clearInterval(state.timer_id);
                state.timer_id = null;
            }
        },

        select_card(card) {
            if (!state.is_game_active || state.is_board_locked) return;
            if (!card || card.dataset.matched === "true") return;
            if (state.selected_cards.includes(card)) return;

            card.classList.add(state.config.active_class);
            state.selected_cards.push(card);

            if (state.selected_cards.length === 2) {
                PassionGame.evaluate_selected_cards();
            }
        },

        evaluate_selected_cards() {
            if (state.selected_cards.length < 2) return;

            const
                [first_card, second_card] = state.selected_cards,
                first_id = first_card ? first_card.dataset.id : null,
                second_id = second_card ? second_card.dataset.id : null
            ;

            if (first_card && second_card && first_id && second_id && first_id === second_id) {
                first_card.dataset.matched = "true";
                second_card.dataset.matched = "true";

                first_card.classList.add(state.config.matched_class);
                second_card.classList.add(state.config.matched_class);

                state.matched_pairs += 1;
                utils.update_matches_display();
                state.selected_cards = [];

                if (state.matched_pairs >= state.config.total_pairs) {
                    PassionGame.complete_game(true);
                }
                return;
            }

            state.is_board_locked = true;

            window.setTimeout(() => {
                for (const card of state.selected_cards) {
                    if (!card) continue;
                    if (card.dataset.matched === "true") continue;
                    card.classList.remove(state.config.active_class);
                }

                state.selected_cards = [];
                if (!state.has_dispatched_result) {
                    state.is_board_locked = false;
                }
            }, FLIP_BACK_DELAY);
        },

        complete_game(did_win) {
            if (state.has_dispatched_result) return;

            state.has_dispatched_result = true;
            state.is_game_active = false;
            state.is_board_locked = true;
            PassionGame.stop_timer();
            utils.set_reload_button_hidden(false);

            const
                event_name = did_win ? "passionGame:win" : "passionGame:lose",
                detail = {
                    time_remaining: state.timer_remaining,
                    matched_pairs: state.matched_pairs
                },
                target = state.config.card_container || document
            ;

            target.dispatchEvent(new CustomEvent(event_name, { detail }));

        },

        handle_defeat() {
            if (state.has_dispatched_result) return;
            PassionGame.complete_game(false);
        },

        restart_game() {
            PassionGame.stop_timer();
            state.is_game_active = false;
            state.is_board_locked = false;
            state.has_dispatched_result = false;
            PassionGame.start_game();
        },

        rebuild() {
            utils.render_cards();
            state.selected_cards = [];
            state.matched_pairs = 0;
            state.is_board_locked = false;
            state.is_game_active = false;
            state.has_dispatched_result = false;
            state.timer_remaining = state.base_timer_duration;
            utils.set_reload_button_hidden(false);
            utils.update_timer_display(state.timer_remaining);
            utils.update_matches_display();
        },

        get_state() {
            return {
                timer_remaining: state.timer_remaining,
                timer_duration: state.timer_duration,
                random_cards: state.random_cards.slice(),
                matched_pairs: state.matched_pairs,
                is_game_active: state.is_game_active,
                has_dispatched_result: state.has_dispatched_result
            };
        }
    };

    window.PassionGame = PassionGame;
}());
