(function () {
    // Config
    const canvas = document.getElementById("fall_game");
    const ctx = canvas.getContext("2d");
    const start_btn = document.getElementById("start_btn");

    const DEBUG = canvas.dataset.debug === "true"; // modo debug

    const COLS = 7; // columnas

    let CAPTURE_R = 30; // radio anillo de captura (stroke)
    let BALL_R = CAPTURE_R; // radio balón (relleno), tamaño del balón
    let CAPTURE_Y = Math.round(canvas.height * 0.38); // línea de captura
    let TIMER_POS = [canvas.width / 2, CAPTURE_Y + 80]; // posición cronometro

    const RING_STROKE = 3; // grosor anillo
    const COL_SPACE = 10; // espacio entre columnas
    const COLOR_LETTER = "#FF78D9"; // color letras
    const TOTAL_TIME = 30; // tiempo total en segundos

    let LETTER_SCALE = 5.6; // escala letras
    let LETTER_X_OFFSET = 31; // desplazamiento x letras
    let LETTER_Y = CAPTURE_Y + 120; // desplazamiento y letras

    let FONT_SIZES = {
        levelLabel: 28,
        levelNumber: 64,
        levelNumberX: 45,
        levelNumberY: 52,
        timer: 42
    }; // tamaños fuentes

    // end Config

    // Niveles
    const LEVELS = [
        { id: 1, hit_margin: 100, ball_speed: 240, emit_ms: 1000 },
        { id: 2, hit_margin: 60, ball_speed: 280, emit_ms: 900 },
        { id: 3, hit_margin: 20, ball_speed: 340, emit_ms: 500 },
    ];
    // end Niveles

    // Estado
    let currentLevelIdx = 0; // 0 → nivel 1

    let HIT_MARGIN = LEVELS[currentLevelIdx].hit_margin;
    let BALL_SPEED = LEVELS[currentLevelIdx].ball_speed;
    let BALL_EMIT_INTERVAL = LEVELS[currentLevelIdx].emit_ms;

    const balls = []; // { col_index, x, y, vy, overlapping }
    const passed_by_col = Array(COLS).fill(0); // indicadores DEV
    const caught_by_col = Array(COLS).fill(0); // indicadores DEV
    const col_active = Array(COLS).fill(0); // 0 = libre, 1 = ocupado

    let last_time = performance.now(); // ms
    let next_emit_at = performance.now() + BALL_EMIT_INTERVAL; // ms
    let is_playing = false;

    let time_left = TOTAL_TIME; // en segundos
    let last_tick_time = performance.now(); // ms
    let gameEnded = false; // evita logs repetidos y corta la emisión
    let rafId = null; // id del requestAnimationFrame activo
    // end Estados

    // Asset del balón
    const
        BALL_IMG_SRC = canvas.dataset.ballSrc || "img/ball/trionda.webp",
        ball_img = new Image()
    ;

    let ball_img_ready = false;
    ball_img.decoding = "async";
    // Si más adelante sirves desde CDN, esto evitará problemas de CORS al dibujar:
    ball_img.crossOrigin = "anonymous";

    ball_img.onload = () => {
        ball_img_ready = true;
        if (DEBUG) console.log("[ball] loaded:", BALL_IMG_SRC);
    };

    ball_img.onerror = (ev) => {
        console.warn("[ball] FAIL loading:", BALL_IMG_SRC, ev);
    };

    ball_img.src = BALL_IMG_SRC;
    // end inclusion del balón

    // Aplicar nivel
    function applyLevel(idx) {
        currentLevelIdx = Math.max(0, Math.min(idx, LEVELS.length - 1));
        const L = LEVELS[currentLevelIdx];

        // aplica parámetros del nivel
        HIT_MARGIN = L.hit_margin;
        BALL_SPEED = L.ball_speed;
        BALL_EMIT_INTERVAL = L.emit_ms;

        // resetea estados de la partida
        balls.length = 0;
        for (let i = 0; i < COLS; i++) {
            col_active[i] = 0;
            passed_by_col[i] = 0;
            caught_by_col[i] = 0;
            lettersFill[i] = 0;
        }

        // reinicia solo el spawner para que arranque rápido en el nuevo nivel
        next_emit_at = performance.now() + 300;

        // seguimos jugando (no mostramos start de nuevo)
        is_playing = true;
    } // end applyLevel


    // Crear el path de las letras
    const path_T = new Path2D("M10.997 0v2.51h-4.15v16.8h-2.67V2.51h-4.18V0z");
    const path_R = new Path2D("M9.955-.002v10.96h-2.34l.11.35 2.57 8.01h-2.75l-3.47-10.87h3.22v-5.94h-4.63v16.81h-2.67V-.002Z");
    const path_I = new Path2D("M9.065.005v2.51h-3.08v14.29h3.33v2.51h-9.32v-2.51h3.33V2.515H.255V.005z");
    const path_O = new Path2D("M10.2 0v19.3H0V0ZM2.6 16.8h5V2.5h-5z");
    const path_N = new Path2D("M10.28.002v19.31H7.63v-16.8H2.66v16.8H0V.002Z");
    const path_D = new Path2D("M9.035-.002v1.65h1.57v15.69h-1.47v1.97H.005V-.002Zm-6.37 16.8h5.27V2.508h-5.27z");
    const path_A = new Path2D("M10.275-.002v19.31h-2.65v-8.44h-4.96v8.44H.005V-.002Zm-7.61 8.36h4.96v-5.85h-4.96z");

    const lettersArray = [path_T, path_R, path_I, path_O, path_N, path_D, path_A];
    const lettersFill = Array(lettersArray.length).fill(0);

    // Función para dibujar letras según el path
    function drawLetter(path, posx, posy, index) {
        ctx.save();
        ctx.translate(posx, posy);
        ctx.scale(LETTER_SCALE, LETTER_SCALE);

        ctx.strokeStyle = COLOR_LETTER;
        ctx.lineWidth = 0.5;
        ctx.stroke(path);

        if (lettersFill[index] === 1) {
            ctx.fillStyle = COLOR_LETTER;
            ctx.fill(path);
        }

        ctx.restore();
    }

    // Utilidades
    // Random entre min y max
    function rand_between(min, max) { return Math.floor(Math.random() * (max - min + 1)) + min; }

    // Start Loop
    function startLoop() {
        if (!rafId) {
            rafId = requestAnimationFrame(tick);
        }
    } // end startLoop

    // Stop Loop
    function stopLoop() {
        if (rafId) {
            cancelAnimationFrame(rafId);
            rafId = null;
        }
    } // end stopLoop

    // Emitir evento, fallgame:win | fallgame:lose
    function emitGameEvent(type, detail) {
        const ev = new CustomEvent(`fallgame:${type}`, { detail });
        window.dispatchEvent(ev);
    } // end emitGameEvent

    // Fin de la partida, "win" | "lose"
    function endGame(result) {
        if (gameEnded) return;

        // cambio de estado para finalizar
        gameEnded = true;
        is_playing = false;

        emitGameEvent(result, {
            level: LEVELS[currentLevelIdx].id,
            captured: caught_by_col.reduce((a, b) => a + b, 0),
            missed: passed_by_col.reduce((a, b) => a + b, 0),
            time_left,
            total_cols: COLS
        });

        stopLoop();
    }

    // Creación de columnas
    function getColX(i) {
        const
            total_width = COLS * (CAPTURE_R * 2) + (COLS - 1) * COL_SPACE,
            start_x = (canvas.width - total_width) / 2 + CAPTURE_R
        ;

        return start_x + i * ((CAPTURE_R * 2) + COL_SPACE);
    } // end getColX

    // Dibujar indicador de balones capturados
    function drawDebugBottomLeft() {
        if (!DEBUG) return;

        const margin = 12;
        const lineH = 16;
        let lines = [
            `balls: ${balls.length}`,
            `caught: ${caught_by_col.reduce((a, b) => a + b, 0)}`,
            `missed: ${passed_by_col.reduce((a, b) => a + b, 0)}`,
            `NIVEL: ${LEVELS[currentLevelIdx].id}`
        ];

        // chequeo de imagen cargada
        if (!ball_img_ready) {
            lines.push("ball_img: NOT LOADED (revise ruta/MIME)");
            lines.push(BALL_IMG_SRC);
        }

        ctx.save();
        ctx.font = "14px monospace";
        ctx.fillStyle = "#F55";
        ctx.textAlign = "left";
        ctx.textBaseline = "top";

        // dibuja todas las líneas, alineadas abajo-izquierda
        const y0 = canvas.height - margin - lineH * lines.length;
        for (let i = 0; i < lines.length; i++) {
            ctx.fillText(lines[i], margin, y0 + i * lineH);
        }
        ctx.restore();
    } // end drawDebugBottomLeft

    // Generación de balones
    function spawnBall() {
        if (!is_playing) return;

        const now = performance.now();
        if (now < next_emit_at) return;

        // busca columnas libres
        const freeCols = [];
        for (let i = 0; i < COLS; i++) {
            if (col_active[i] === 0) freeCols.push(i);
        }
        if (freeCols.length === 0) return; // ninguna libre

        // elige una columna libre al azar
        const col_index = freeCols[rand_between(0, freeCols.length - 1)];

        balls.push({
            col_index,
            x: getColX(col_index),
            y: -BALL_R * 2,
            vy: BALL_SPEED,
            overlapping: false
        });

        col_active[col_index] = 1; // ahora está ocupada
        next_emit_at = now + BALL_EMIT_INTERVAL;
    }

    // Dibujar indicador de balones capturados
    function drawCapturedIndicator(ctx, captured, total, x, y, options = {}) {
        const {
            font = "700 40px adidasFG",
            color = "#FFF",
            spacing = -6,
            align = "right"
        } = options;

        ctx.save();
        ctx.font = font;
        ctx.fillStyle = color;
        ctx.textBaseline = "top";

        // armar el string
        const text = `${captured} /${total}`;
        const chars = text.split("");

        // calcular ancho total con spacing
        let totalWidth = 0;
        chars.forEach((ch, i) => {
            totalWidth += ctx.measureText(ch).width;
            if (i < chars.length - 1) totalWidth += spacing;
        });

        // ajustar x inicial según align
        let startX = x;
        if (align === "center") startX = x - totalWidth / 2;
        if (align === "right") startX = x - totalWidth;

        // dibujar carácter por carácter
        let currentX = startX;
        for (let i = 0; i < chars.length; i++) {
            const ch = chars[i];
            ctx.fillText(ch, currentX, y);
            currentX += ctx.measureText(ch).width + spacing;
        }

        ctx.restore();
    } // end drawCapturedIndicator

    // Dibujar indicador de nivel
    function drawLevelIndicator() {
        const levelNumber = String(LEVELS[currentLevelIdx].id).padStart(2, "0");

        // "NIVEL"
        ctx.font = `700 ${FONT_SIZES.levelLabel}px adidasFG`;
        ctx.fillStyle = "#FFF";
        ctx.textAlign = "left";
        ctx.textBaseline = "top";
        ctx.fillText("NIVEL", FONT_SIZES.levelNumberX, 25);

        // Número de nivel
        ctx.font = `700 ${FONT_SIZES.levelNumber}px adidasFG`;
        ctx.fillText(levelNumber, FONT_SIZES.levelNumberX, FONT_SIZES.levelNumberY);
    } // end drawLevelIndicator

    // Recalcular tamaños de captura
    function recalcSizes(isMobile) {
        if (isMobile) {
            CAPTURE_Y = Math.round(canvas.height * 0.45);

            const marginX = 20;
            const avail = canvas.width - 2 * marginX - (COLS - 1) * COL_SPACE;
            const diam = Math.max(14, Math.floor(avail / COLS * 0.75));

            CAPTURE_R = Math.floor(diam / 2);
            BALL_R = CAPTURE_R;

            TIMER_POS = [canvas.width / 2, CAPTURE_Y + Math.round(2.6 * CAPTURE_R)];

            const k = CAPTURE_R / 30;

            LETTER_SCALE = 5.6 * k;
            LETTER_X_OFFSET = Math.round(31 * k);
            LETTER_Y = CAPTURE_Y + Math.round(120 * k);

            FONT_SIZES = {
                levelLabel: 32,
                levelNumber: 76,
                levelNumberX: 40,
                levelNumberY: 52,
                timer: 64
            };
        } else {
            CAPTURE_R = 28;
            BALL_R = CAPTURE_R;
            CAPTURE_Y = Math.round(canvas.height * 0.38);
            TIMER_POS = [canvas.width / 2, CAPTURE_Y + 80];
            LETTER_SCALE = 5.6;
            LETTER_X_OFFSET = 31;
            LETTER_Y = CAPTURE_Y + 120;
        }
    } // end recalcSizes

    // Redimensionar canvas
    function resizeCanvas() {
        const isMobile = window.matchMedia("(max-width: 640px)").matches;
        const target = isMobile ? { w: 640, h: 900 } : { w: 1280, h: 595 };

        canvas.width = target.w;
        canvas.height = target.h;

        recalcSizes(isMobile);
    } // end resizeCanvas

    // Update / Draw
    function tick(now) {
        const dt = Math.min(0.033, (now - last_time) / 1000); // clamp 33ms
        last_time = now;

        // Spawn
        spawnBall();

        // Update
        for (let i = balls.length - 1; i >= 0; i--) {
            const b = balls[i];
            b.y += b.vy * dt;

            // overlapping con su anillo de captura
            b.overlapping =
                b.y >= CAPTURE_Y - (CAPTURE_R + HIT_MARGIN) &&
                b.y <= CAPTURE_Y + (CAPTURE_R + HIT_MARGIN);

            // salió por abajo = pasó la fila (miss)
            if (b.y - BALL_R > canvas.height) {
                passed_by_col[b.col_index]++;
                col_active[b.col_index] = 0;
                balls.splice(i, 1);
            }
        }

        // Update Timer
        if (is_playing && time_left > 0) {
            const elapsed = (now - last_tick_time) / 1000;
            if (elapsed >= 1) {
                time_left = Math.max(0, time_left - Math.floor(elapsed));
                last_tick_time = now;
            }
        } else { last_tick_time = now; }

        // Game Over
        if (!gameEnded && time_left === 0) {
            endGame("lose");
        }

        // Draw
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        // anillos de captura por columna
        ctx.lineWidth = RING_STROKE;
        ctx.strokeStyle = "#FFF";
        ctx.fillStyle = "#000";
        for (let i = 0; i < COLS; i++) {
            const x = getColX(i);
            ctx.beginPath();
            ctx.arc(x, CAPTURE_Y, CAPTURE_R, 0, Math.PI * 2);
            ctx.stroke();

            if (DEBUG) {
                ctx.font = "14px monospace";
                ctx.textAlign = "center";
                ctx.textBaseline = "top";
                ctx.fillText(String(passed_by_col[i]), x, CAPTURE_Y + CAPTURE_R + 6);
            }
        }

        // balones
        for (const b of balls) {
            const d = BALL_R * 2;

            if (ball_img_ready) {
                // Imagen escalada al diámetro del balón
                ctx.imageSmoothingEnabled = true;
                ctx.imageSmoothingQuality = "high";
                ctx.drawImage(ball_img, b.x - BALL_R, b.y - BALL_R, d, d);

                // Overlay rojo en DEBUG cuando está en vecindad (opcional, mantiene tu señal visual)
                if (DEBUG && b.overlapping) {
                    ctx.globalAlpha = 0.35;
                    ctx.beginPath();
                    ctx.arc(b.x, b.y, BALL_R, 0, Math.PI * 2);
                    ctx.fillStyle = "red";
                    ctx.fill();
                    ctx.globalAlpha = 1;
                }
            } else {
                // Fallback mientras carga la imagen
                ctx.beginPath();
                ctx.arc(b.x, b.y, BALL_R, 0, Math.PI * 2);
                ctx.fillStyle = (DEBUG && b.overlapping) ? "red" : "#000";
                ctx.fill();
            }

            // indicador visual dev al cruzar la zona
            if (DEBUG && b.overlapping) {
                ctx.beginPath();
                ctx.arc(b.x, CAPTURE_Y, CAPTURE_R + 6, 0, Math.PI * 2);
                ctx.setLineDash([6, 6]);
                ctx.stroke();
                ctx.setLineDash([]);
            }
        }

        // Dibujar las letras
        lettersArray.forEach((letter, i) => {
            drawLetter(letter, getColX(i) - LETTER_X_OFFSET, LETTER_Y, i);
        });


        // Indicador top right
        ctx.font = "700 75px adidasFG";
        ctx.fillStyle = "#FFF";
        ctx.textAlign = "right";
        ctx.textBaseline = "top";

        // número de capturados
        const captured = caught_by_col.reduce((a, b) => a + b, 0);
        drawCapturedIndicator(
            ctx, captured,
            COLS,
            canvas.width - 20, 20,
            {
                font: "700 60px adidasFG",
                color: "#FFF",
                spacing: -4,
                align: "right"
            }
        );

        // Cronómetro
        const minutes = Math.floor(time_left / 60);
        const seconds = time_left % 60;

        const time_str = `${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;

        ctx.font = `700 ${FONT_SIZES.timer}px adidasFG`;
        ctx.fillStyle = "#FFF";
        ctx.textAlign = "center";
        ctx.textBaseline = "middle";

        ctx.fillText(time_str, TIMER_POS[0], TIMER_POS[1]);

        // Nivel
        drawLevelIndicator();

        // Debug
        drawDebugBottomLeft();

        rafId = requestAnimationFrame(tick);
    } // end tick

    // Capturar balón
    function catchBall(b) {
        const idx = b.col_index;
        if (b.vy === 0) return;

        caught_by_col[idx]++;
        b.y = CAPTURE_Y;
        b.vy = 0;
        b.overlapping = false;
        lettersFill[idx] = 1;

        // Revisar si todos los balones fueron capturados
        const allFilled = lettersFill.every(v => v === 1);

        if (allFilled) {
            if (currentLevelIdx < LEVELS.length - 1) {
                applyLevel(currentLevelIdx + 1);
            } else if (!gameEnded) {
                setTimeout(() => endGame("win"), 500);
            }
        }
    } // end catchBall


    /* Input (click/touch sobre el anillo correcto
    *  mientras el balón lo cruza) */
    canvas.addEventListener("pointerdown", (ev) => {
        const
            rect = canvas.getBoundingClientRect(),
            px = (ev.clientX - rect.left) * (canvas.width / rect.width),
            py = (ev.clientY - rect.top) * (canvas.height / rect.height)
        ;

        // detectar columna por anillo
        let col_clicked = -1;
        for (let i = 0; i < COLS; i++) {
            const
                x = getColX(i),
                dx = px - x,
                dy = py - CAPTURE_Y,
                dist = Math.hypot(dx, dy)
            ;
            if (dist <= CAPTURE_R + RING_STROKE * 1.5) {
                col_clicked = i;
                break;
            }
        }

        // click dentro de un anillo
        if (col_clicked !== -1) {
            for (let i = 0; i < balls.length; i++) {
                const b = balls[i];
                if (b.col_index === col_clicked && b.overlapping) {
                    catchBall(b);
                    return;
                }
            }
        }

        // click directamente sobre balón en vecindad
        for (let i = 0; i < balls.length; i++) {
            const
                b = balls[i],
                dx = px - b.x,
                dy = py - b.y,
                dist = Math.hypot(dx, dy)
            ;

            // click dentro del círculo del balón
            if (dist <= BALL_R && b.overlapping) {
                catchBall(b);
                return;
            }
        }
    }); // end Input

    resizeCanvas();
    window.addEventListener("resize", resizeCanvas);
    window.addEventListener("orientationchange", () => setTimeout(resizeCanvas, 100));

    requestAnimationFrame(tick);

    // Iniciar partida
    start_btn.addEventListener("pointerdown", function (e) {
        e.preventDefault();
        if (is_playing) return;

        this.classList.add("hide");

        is_playing = true;
        gameEnded = false;

        time_left = TOTAL_TIME;
        last_tick_time = performance.now();

        // primer balón pronto
        next_emit_at = performance.now() + 60;
        applyLevel(0);

        startLoop();
    }); // end Input
}());

